/******************************************************************************
 gcl.pl version 2.2 13/2/2017 (Pedro Cabalar)

******************************************************************************/ 

:-op(1070,xfx,:=).	% assignment
:- dynamic var/2, 	% var(Variable,Type)
           initval/2,   % initval(Variable,Initialvalue)
           const/2,     % const(Constantname,Value)
	   function/5,  % function(Name/Arity,Type,Arguments,Sentences)  
	   program/1.	% program(Sentences)


:- dynamic optStep/0,optTrace/0,optAll/0,optQuiet/0,
           solNum/1,write_state/1.

%----- main ------------------------------------------------------------------

% retrieves command line arguments Args and calls main(Args)
main :- unix(argv(Argv)),
        ( append(_PrologArgs, [--|AppArgs], Argv), !,
          main(AppArgs)
        ; main(Argv)).

% if no arguments, print help
main([]):- !,
  header,nl,
  write('gcl [options] <filename>'),nl,
  write('  -t  print whole trajectory after each execution'), nl,
  write('  -a  show all nondeterministic solutions'), nl,
  write('  -s  perform a single (random) execution step by step showing'), nl,
  write('      current line and resulting state (disabled by -a or -t)'), nl,
  write('  -q  quiet: supresses information header and messages'), nl.

% otherwise, set options depending on arguments
main(Args):-
	(member('-t',Args),!, asserta(optTrace); true),
	(member('-a',Args),!, asserta(optAll); true),
	(member('-q',Args),!, asserta(optQuiet); true),
	(\+ optTrace, \+ optAll,!,
  	  (member('-s',Args),!, asserta(optStep); true)
        ; true),	
	append(_,[Fname|[]],Args),
	(optQuiet,! ; header),
	
	% Parse and create temporal prolog file
	concat_atom([Fname,'.pl'],TmpFile),
	concatall(['gcl2pl < ',Fname,' > ',TmpFile],Command),		
	(shell(Command),!;true),
	
	loadfile(TmpFile),
	program(P),

	% Some semantic check for initial values
	checkinitvals,
	
	% look for undeclared vars and declare them as integer
	getids(P,Ids),
	undeclaredvars(Ids),
	
	% remove temporal file 
	concatall(['rm ',TmpFile],Command2),
	shell(Command2),

	(optQuiet,! ; write('executing '),write(Fname),write(' ...'),nl),

	init(S),debug(1,S),
	( optAll,!,
	  repeat, (exec([S],P,Tn),write_solution(Tn),fail ; ! )
	; exec([S],P,Tn),
	  ( optStep,!
	  ; optTrace,!,reverse(Tn,T),write_trajectory(T),nl
	  ; Tn=[Sn|_],write_trajectory([Sn]),nl)
	).

header :-
  write('% GCL (version 2.2, Feb 2017)'),nl,
  write('% A small interpreter for Dijkstra''s Guarded Command Languange.'),nl,
  write('% Pedro Cabalar, Dept. Computacion, Univ. of Corunna.'),nl.

%----- LOADING FILE ----------------------------------------------------

loadfile(F):-
	(retractall(var(_,_)),! ; true),
	(retractall(initval(_,_)),! ; true),
	(retractall(const(_,_)),! ; true),
	(retractall(program(_)),! ; true),
	see(F), read_clauses, seen.

read_clauses:-
	read(C),(
	   C=end_of_file,!
	 ; C=(program(P)),!,
	   asserta(program(P)),
	   read_clauses	   	 
	   
	 ; C=vardecl(V,T,I),!,
	   addvar(V,T),
	   (I=[J],!,setinit(V,J); true),
	   read_clauses

	 ; C=constdecl(N,Val),!,
	   addconst(N,Val),
	   read_clauses

	 ; C=fndecl(Name,Type,Args,Ss),
	   length(Args,M),
	   (predefined(Name/M,_,_),!,
	    writelist(['Error: redeclaration of predefined function ',Name/M,'\n']),fail
	   ;function(Name/M,_,_,_,_),!,
	    writelist(['Error: redeclaration of function ',Name/M,'\n']),fail
	   ;findall(X,member(X:_,Args),Xs),
	    findall(W,member(_:W,Args),Ws),
	    assert(function(Name/M,Type,Xs,Ws,Ss))
	   ),
	   read_clauses
	).


% Gets all the non-array identifiers referred in the program. This is done by
% inspecting terms like id(_).

getids(T,As):-
	  T=id(ID),!,As=[ID]
	; functor(T,_,N),
	  ( N=0,!,As=[]
	  ; getids_args(T,N,As)
	  ).

getids_args(_T,0,[]):-!.
getids_args(T,N,As):-
	arg(N,T,Arg),
	getids(Arg,Bs),
	M is N - 1,
	getids_args(T,M,Cs),
	merge_set(Bs,Cs,As).

% var(V,T) stores the database of variables and their types.
% Arrays are stored like var(b, array(0,5,integer) ).
% Latest variables are stored first (local vars in functions, for instance)

addvar(V,T):- asserta(var(V,T)).

addconst(C,V):- const(C,_),!,write('Constant '),write(C),write(' declared twice\n'),fail
                ; assertz(const(C,V)).

undeclaredvars([]):-!.
undeclaredvars([V|Vs]):-(var(V,_),!;asserta(var(V,integer))),undeclaredvars(Vs).

retractvars([]):-!.
retractvars([V|Vs]):-retract(var(V,_)),!,retractvars(Vs).

setinit(X,Es):-
	(retractall(initval(X,_)),!;true),asserta(initval(X,Es)).

% Some semantic checkings about initializations
checkinitvals:- repeat, (initval(X,V), checkinit(X,V),fail; !).

checkinit(X,Es):-
	is_list(Es),!,
	var(X,array(N1,N2,T)),!,
	eval([],N1,M1),
	(number(M1),!; write('Left index in array declaration cannot evaluated.\n'),fail),
	eval([],N2,M2),
	(number(M2),!; write('Right index in array declaration cannot evaluated.\n'),fail),
	N is M2-M1+1,
	( length(Es,N),!
	; write('Array '),write(X),write(' requires '),write(N),
	  write(' values for its initialization.'),nl,fail
	),
        retractall(var(X,array(N1,N2,T))),
        assertz(var(X,array(M1,M2,T))).

checkinit(X,E):-
	  is_list(E),!, write('Variable '),write(X),
	  write(' requires an atomic value.'),nl,fail
	  
	; (retractall(initval(X,_)),!;true),asserta(initval(X,E)).


%----- Evaluation -----------------------------------------------------------

% eval(S,E,V) evaluation of expression E at state S yields value V

eval(S,call(F,Args),V)  :- !,
			   length(Args,N),
			   ( predefined(F/N,_,_),!,
			     evalargs(S,Args,As),
			     predefined(F/N,As,V)
			   ; function(F/N,Type,Vars,Types,Prog),!,
	   	             evalargs(S,Args,Vs),
			     ( maplist(checktype,Types,Vs),!,
			       maplist(addvar,[result|Vars],[Type|Types]),
			       assignvars(Vars,Vs,S0),
			       append([(result,undef)|S0],S,S1),
			       exec([S1],Prog,S2),S2=[Sn|_],
			       (member((result,V),Sn),!; V=undef),
			       retractvars([result|Vars]),!
		             ; V=undef % wrong argument type: function is undefined
			     )
			   ; writelist(['Error: function ',F/N,' not declared.\n']),fail
			   ).

eval(S,cor(X,Y),V)	:- !,eval(S,X,Vx),eval(S,Y,Vy),cor(Vx,Vy,V).
eval(S,cand(X,Y),V)	:- !,eval(S,X,Vx),eval(S,Y,Vy),cand(Vx,Vy,V).
eval(S,or(X,Y),V)	:- !,(defined(S,X,Y,Vx,Vy),!,cor(Vx,Vy,V); V=undef).
eval(S,and(X,Y),V)	:- !,(defined(S,X,Y,Vx,Vy),!,cand(Vx,Vy,V); V=undef).
eval(S,not(X),V)	:- !,(defined(S,X,Vx),!,not(Vx,V); V=undef).
eval(S,eq(X,Y),V)	:- !,(defined(S,X,Y,Vx,Vy),!,truth(Vx=Vy,V); V=undef).
eval(S,neq(X,Y),V)	:- !, eval(S, not(eq(X,Y)),V).
eval(S,gt(X,Y),V)	:- !,(defined(S,X,Y,Vx,Vy),!,truth(Vx>Vy,V); V=undef).
eval(S,lt(X,Y),V)	:- !,(defined(S,X,Y,Vx,Vy),!,truth(Vx<Vy,V); V=undef).
eval(S,geq(X,Y),V)	:- !,(defined(S,X,Y,Vx,Vy),!,truth(Vx>=Vy,V); V=undef).
eval(S,leq(X,Y),V)	:- !, eval(S,geq(Y,X),V).
eval(S,plus(X,Y),V)	:- !,(defined(S,X,Y,Vx,Vy),!,V is Vx+Vy; V=undef).
eval(S,minus(X,Y),V)	:- !,(defined(S,X,Y,Vx,Vy),!,V is Vx-Vy; V=undef).
eval(S,- X,V)		:- !,(defined(S,X,Vx),     !,V is -Vx; V=undef).
eval(S,times(X,Y),V)	:- !,(defined(S,X,Y,Vx,Vy),!,V is Vx*Vy; V=undef).
eval(S,div(X,Y),V)	:- !,(defined(S,X,Y,Vx,Vy),Vy \== 0,!,V is Vx/Vy
                             ;V=undef).
eval(S,mod(X,Y),V)	:- !,(defined(S,X,Y,Vx,Vy),Vy \== 0,!,V is Vx mod Vy
                             ;V=undef).
eval(_S,false,false)	:- !.	
eval(_S,true,true)	:- !.
eval(_S,constant(N),V)	:- !,const(N,V).
eval(_S,K,K)		:- number(K),!.

%eval(_S,ch(K),ch(K)):- !.

eval(S,arr(X,[I]),Vx)	:- 
	!, var(X,array(N1,N2,_T)), eval(S,I,Vi),
	( Vi >= N1, N2 >= Vi,!,J is Vi-N1,
  	   ( member((X,L),S),!,nth0(J,L,Vx)
	   ; Vx=undef
	   )
	; Vx=undef
	).		
eval(S,id(X),V)		:- member((X,V),S),!; V=undef.

cor(true ,_ ,true ):-!.
cor(false,V ,V    ):-!.
cor(_    ,_ ,undef):-!.

cand(false ,_ ,false ):-!.
cand(true  ,V ,V    ):-!.
cand(_     ,_ ,undef):-!.

not(true ,false).
not(false,true ).

defined(S,X,Y,Vx,Vy):- eval(S,X,Vx), Vx \== undef, eval(S,Y,Vy), Vy \== undef.
defined(S,X,Vx)     :- eval(S,X,Vx), Vx \== undef.

truth(F,true):- F,!.
truth(_F,false).

evalargs(_,[],[]):-!.
evalargs(S,[E|Es],[V|Vs]):-eval(S,E,V),evalargs(S,Es,Vs).

checktype(boolean,true):-!.
checktype(boolean,false):-!.
checktype(integer,V) :- !,number(V).
checktype(char,V)    :- !,number(V), 255 >= V, V>=0.

assignvars([],[],[]):-!.
assignvars([X|Xs],[V|Vs],[(X,V)|Ps]):-!,assignvars(Xs,Vs,Ps).

predefined(sqrt/1,[X],V) :- number(X),X>=0,!,V is floor(sqrt(X)); V=undef.
predefined(sin/1,[X],V)  :- number(X),!,V is floor(sin(X)); V=undef.
predefined(cos/1,[X],V)  :- number(X),!,V is floor(cos(X)); V=undef.
predefined(tan/1,[X],V)  :- number(X),C is cos(X),C\=0,!,V is floor(tan(X)); V=undef.
predefined(exp/1,[X],V)  :- number(X),!,V is floor(exp(X)); V=undef.
predefined(exp/2,[X,Y],V)  :- number(X),number(Y),!,V is X**Y; V=undef.
predefined(log/1,[X],V)  :- number(X),X>=0,!,V is floor(log(X)); V=undef.

%----- Execution ------------------------------------------------------------

% exec(T, A, Tn) T=current trajectory, A=sentence, Tn=resulting trajectory
exec(T, [A|As], Tn):-
	!,exec(T,A,T1),exec(T1,As,Tn).
exec(T, [], T):-!.

exec(_T, abort, _):-!,write('Program aborted: abort'),nl,fail.
exec(T, skip, T)  :-!.
exec([S|Ss], (N:As := Bs),[Sn,S|Ss]):- 
	exec_ass(S,As,Bs,S,Sn), debug(N,Sn).
exec([S|Ss], if(Bs), Tn) :-
	eval_conds(S,Bs,Bs2),
	( member((undef -> _),Bs2),!,
	  write('Program aborted: undefined guard.'),nl,fail
	; execTrueBranch([S|Ss],Bs2,Tn)
	).

exec([S|Ss], do(Bs), Tn) :-
	eval_conds(S,Bs,Bs2),
	( member((undef -> _),Bs2),!,
	  write('Program aborted: undefined guard.'),nl,fail
	; execTrueBranch([S|Ss],Bs2,T1), exec(T1,do(Bs),Tn)
	; \+ member((true -> _),Bs2), !, Tn=[S|Ss]
	).

exec([S|Ss], assert(Line,E), [S|Ss]) :-
	eval(S,E,V),
	( V=true,!
	; V=false, 
	  write('Line '),write(Line),
	  write(': violated assertion.'),nl,fail
	).

/** exec_ass(S,Xs,Es,Si,Sn) executes assignment (Xs := Es) where Xs is a list 
  * of variables and Es a list of expressions. S is the originial state where 
  * expressions are valuated, state Si stores the accumulated changes and 
  * Sn the final state.
  **/

exec_ass(_S,[],[],Si,Si):-!.
exec_ass(S,[arr(X,[I])|Xs],[E|Es],Si,Sn):-
	!, var(X,array(N1,N2,T)),
	eval(S,I,Vi), J is Vi-N1,
	(Vi>=N1,N2>=Vi,!
	; write('Program aborted: index '), write(Vi), 
	  write(' out of range for '),write(X),write('.'),nl,fail
	),
	eval(S,E,Ve),checkdefined(X,T,Ve),
	append(Pre,[(X,L)|Suf],Si),
	setnth0(J,L,Ve,L2),
	append(Pre,[(X,L2)|Suf],S2),
	exec_ass(S,Xs,Es,S2,Sn).
exec_ass(S,[id(X)|Xs],[E|Es],Si,Sn):-
	var(X,T),
	eval(S,E,Ve),checkdefined(X,T,Ve),
	append(Pre,[(X,_)|Suf],Si),
	append(Pre,[(X,Ve)|Suf],S2),
	exec_ass(S,Xs,Es,S2,Sn).

% Given a list of guarded branches, replaces all the conditions 
% by their truth in state S

eval_conds(_S,[],[]):-!.
eval_conds(S,[(C -> B)|Bs],[(Vc -> B)|Ds]):-
	eval(S,C,Vc),
	eval_conds(S,Bs,Ds).

execTrueBranch(T,Bs,Tn):-
	optAll,!,
	member((true -> B),Bs), exec(T,B,Tn)
      ; bagof(B, member((true -> B),Bs),TrueBs),
  	length(TrueBs,N),X is random(N),nth0(X,TrueBs,Branch),
  	exec(T,Branch,Tn).

/** checkdefined(X,T,V)
  *  X - variable, T - type, V - expression value 
  * checks whether V is a correct value for variable X of type T.
  */

checkdefined(_X,integer,V) :- number(V),!.
checkdefined(_X,boolean,true) :- !.
checkdefined(_X,boolean,false) :- !.
checkdefined(_X,char,N) :- number(N), 255 >= N, N>=0,!.
checkdefined(X,T,V):-
	write('Program aborted: '), 
	write(X), write(' is assigned a value '), 
	write(V), write(' out of range ('),
	write(T), write(').'), nl,fail.

debug(N,S):-
  optStep,!,write(N),write(': '),def_write_state(S),nl,get0(_).
debug(_,_).

%--- Generates initial state S
init(S):- bagof((X,V),initvalue(X,V),S).

initvalue(X,V) :- var(X,T),getInitVal(X,T,V).

getInitVal(X,array(N1,N2,_T),V):- !,
	( initval(X,V0),!,evalAll(V0,V)
	; N is N2-N1+1,repeatlist(N,undef,V)
	).

getInitVal(X,_T,V):- initval(X,V),! ; V=undef.

evalAll([],[]):-!.
evalAll([X|Xs],[Y|Ys]):- eval([],X,Y),evalAll(Xs,Ys).

solNum(0).

%---- Write a solution (for option -a)
write_solution(T):-
	solNum(N),retractall(solNum(_)), M is N+1,asserta(solNum(M)),
	write('Solution '),write(M),write(':'),nl,
	(optTrace,!,reverse(T,Tn),write_trajectory(Tn); 
	 T=[S|_],write_trajectory([S])
	).

%---- Some utilities ---------------------------------------------------------
	
% setnth0(I,L1,V,L2) sets I-th position of list L1 to value V returning list L2
setnth0(I,L1,V,L2):- setnth0(I,0,L1,V,L2).

setnth0(_I,_J,[],_V,[]):-!.
setnth0(_I,_J,[],_V,[]):-!.

setnth0(I,I,[_X|L],V,[V|L]):-!.
setnth0(I,J,[X|L],V,[X|L2]):-
	K is J+1, setnth0(I,K,L,V,L2).
	

write_trajectory([]):-!,nl.
write_trajectory([A|As]):-
	( write_state(A),! 
	; def_write_state(A) % default display routine
	),
	nl,write_trajectory(As).

%--- Generates a list with N repetitions of element X
repeatlist(0,_X,[]):-!.
repeatlist(N,X,[X|L]):-M is N-1, repeatlist(M,X,L).

write_var(X,V):-
	write(X),write('='),
	(
	  V=undef,!,write(V)
	; var(X,array(_,_,char)),!,put(0'"),put_chars(V),put(0'")
	; var(X,char),!,put(0''),put(V),put(0'')
	; write(V)
	).

def_write_state([]):-!.
def_write_state([(X,V)]):-!,write_var(X,V).
def_write_state([(X,V)|S]):- write_var(X,V),write(', '),def_write_state(S).

put_chars([]):- !.	
put_chars([C|Cs]):- put(C),put_chars(Cs).

concatall([],''):-!.
concatall([A|As],C):-
	concatall(As,B),concat_atom([A,B],C).

writelist([]).
writelist([X|Xs]):-write(X),writelist(Xs).
